// IGCXXX.C
//
// Example implementation of a DLL implementing the IGC application
// programming interface for data transfer, conversion, and validation.
//
// Copyright (c) 2000-2001 Marc Ramsey. All rights reserved.
//
// http://www.ranlog.com/ramsey/
//
// This software is provided as is with no warranties of any kind, 
// including the warranties of design, merchantibility and fitness 
// for a particular purpose, noninfringement, or arising from a 
// course of dealing, usage or trade practice.  This software is 
// provided with no support and without any obligation to assist in 
// its use, correction, modification, or enhancement.
//
// Permission to use, copy, or modify this software for any purpose 
// is hereby granted without fee, provided the above notices are 
// retained on all copies of the source code.
 
#include "stdafx.h"
#include "resource.h"
#include <stdarg.h>
#include <string.h>
#include <commctrl.h>

// Define string returned by IdentifyDLL.  Replace
// with appropriate string.
#define IDENTIFY_DLL "XXX|Acme Instruments|XL 100, 200|2.0|XL1,XL2|XL1,XL2,IGC"

// Define the time interval in milliseconds required between 
// calls to KeepAwakeFR. Use 0 if the calls are not needed.
#define KEEP_AWAKE_INTERVAL 5000
 
// Comment out the following line if the SerialOptionsDLL
// function is not required.
#define USE_SERIAL_OPTIONS

// Define the default baud rate for the flight recorder
#define DEFAULT_BAUD_RATE CBR_19200
#define DEFAULT_BAUD_RATE_BTN IDC_LS19200

// Comment out the following line if the ConvertLog
// function is not required.
#define USE_CONVERT_LOG

static HANDLE moduleHandle;
static HWND topWindowHandle;
static BOOL quietModeFlag;
static HANDLE commHandle;

#ifdef USE_SERIAL_OPTIONS
static LPTSTR serialOptions;
static DWORD serialOptionsSize;
#endif // USE_SERIAL_OPTIONS

// ErrorBox
//
// Display the supplied error message string in a 
// modal dialog box.
//
static void ErrorBox(LPCTSTR message)
{
	MessageBox(topWindowHandle, (LPCTSTR)message, "Error", MB_OK | MB_ICONEXCLAMATION );
}

// LastErrorBox
//
// Display the last detected error in a modal dialog
//
static void LastErrorBox(LPCTSTR arg, ...)
{
	va_list marker;
	LPVOID lpMsgBuf;

	va_start(marker, arg);

	//
	// Format error message
	//
	FormatMessage( 
		FORMAT_MESSAGE_ALLOCATE_BUFFER | 
		FORMAT_MESSAGE_FROM_SYSTEM,
		NULL,
		GetLastError(),
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
		(LPTSTR) &lpMsgBuf,
		0,
		&marker 
	);

	//
	// Display the dialog
	//
	ErrorBox((LPCTSTR)lpMsgBuf);
	// Free the buffer.
	LocalFree( lpMsgBuf );
	va_end(marker);
}

// CheckSerialPort
//
// Check if serial port is open.  If so, return TRUE,
// otherwise display an error message (if error is TRUE)
// and return FALSE.
//
static BOOL CheckSerialPort(BOOL error)
{
	if (commHandle == INVALID_HANDLE_VALUE) {
		if (error)
			ErrorBox("No serial port connection\n");
		return FALSE;
	} else
		return TRUE;
}

// SerialOptionsProc
//
// Dialog procedure for SerialOptions
//
#ifdef USE_SERIAL_OPTIONS
static CALLBACK SerialOptionsProc(HWND hwndDlg, UINT message, WPARAM wParam, LPARAM lParam) 
{
	HWND hwndOwner; 
	RECT rc, rcDlg, rcOwner;
	int baudRateBtn = DEFAULT_BAUD_RATE_BTN;

	switch (message) {
	case WM_INITDIALOG:
		//
		// Initializing, get owner window and its rectangle
		//
		if ((hwndOwner = GetParent(hwndDlg)) == NULL)
			hwndOwner = GetDesktopWindow(); 
		GetWindowRect(hwndOwner, &rcOwner); 
		GetWindowRect(hwndDlg, &rcDlg); 
		CopyRect(&rc, &rcOwner); 

		//
		// Center dialog box above owner window
		//
		OffsetRect(&rcDlg, -rcDlg.left, -rcDlg.top); 
		OffsetRect(&rc, -rc.left, -rc.top); 
		OffsetRect(&rc, -rcDlg.right, -rcDlg.bottom); 
		SetWindowPos(hwndDlg, HWND_TOP, 
					 rcOwner.left + (rc.right / 2), 
					 rcOwner.top + (rc.bottom / 2), 
					 0, 0, SWP_NOSIZE);

		//
		// Set initial state of baud rate radio buttons
		//
		if (strcmp(serialOptions, "4800") == 0)
			baudRateBtn = IDC_LS4800;
		else if (strcmp(serialOptions, "9600") == 0)
			baudRateBtn = IDC_LS9600;
		else if (strcmp(serialOptions, "19200") == 0)
			baudRateBtn = IDC_LS19200;
		else if (strcmp(serialOptions, "38400") == 0)
			baudRateBtn = IDC_LS38400;
		else if (strcmp(serialOptions, "57600") == 0)
			baudRateBtn = IDC_LS57600;
		else if (strcmp(serialOptions, "115200") == 0)
			baudRateBtn = IDC_LS115200;
		CheckRadioButton(hwndDlg, IDC_LS4800, IDC_LS115200, baudRateBtn);

		//
		// TODO: if there are additional option controls, initialize them
		//

		//
		// Dialog initialization complete
		//
		return TRUE;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDOK:
			//
			// OK button pressed, determine which baud rate button is set
			//
			if (IsDlgButtonChecked(hwndDlg, IDC_LS4800) == BST_CHECKED)
				strncpy(serialOptions, "4800", serialOptionsSize);
			else if (IsDlgButtonChecked(hwndDlg, IDC_LS9600) == BST_CHECKED)
				strncpy(serialOptions, "9600", serialOptionsSize);
			else if (IsDlgButtonChecked(hwndDlg, IDC_LS19200) == BST_CHECKED)
				strncpy(serialOptions, "19200", serialOptionsSize);
			else if (IsDlgButtonChecked(hwndDlg, IDC_LS38400) == BST_CHECKED)
				strncpy(serialOptions, "38400", serialOptionsSize);
			else if (IsDlgButtonChecked(hwndDlg, IDC_LS57600) == BST_CHECKED)
				strncpy(serialOptions, "57600", serialOptionsSize);
			else if (IsDlgButtonChecked(hwndDlg, IDC_LS115200) == BST_CHECKED)
				strncpy(serialOptions, "115200", serialOptionsSize);
			serialOptions[serialOptionsSize - 1] = 0;
			serialOptionsSize = strlen(serialOptions);

			//
			// TODO: insert code here for any additional options
			//

			// Dialog complete
			EndDialog(hwndDlg, wParam);
			return TRUE;
		case IDCANCEL:
			//
			// Dialog has been cancelled, return 0 for size
			//
			serialOptionsSize = 0;
			EndDialog(hwndDlg, wParam);
			return TRUE;
		}
	}
	return FALSE;
}
#endif // USE_SERIAL_OPTIONS

static BOOL progressCancelled;
static HWND progressDialog;

static BOOL CALLBACK ProgressProc(HWND hwndDlg, UINT message, WPARAM wParam, LPARAM lParam) 
{
	HWND hwndOwner; 
	RECT rc, rcDlg, rcOwner; 

    switch (message) { 
    case WM_INITDIALOG: 

        // Get the owner window and dialog box rectangles. 
 
        if ((hwndOwner = GetParent(hwndDlg)) == NULL) {
            hwndOwner = GetDesktopWindow(); 
        }

        GetWindowRect(hwndOwner, &rcOwner); 
        GetWindowRect(hwndDlg, &rcDlg); 
        CopyRect(&rc, &rcOwner); 
 
         // Offset the owner and dialog box rectangles so that 
         // right and bottom values represent the width and 
         // height, and then offset the owner again to discard 
         // space taken up by the dialog box. 
 
        OffsetRect(&rcDlg, -rcDlg.left, -rcDlg.top); 
        OffsetRect(&rc, -rc.left, -rc.top); 
        OffsetRect(&rc, -rcDlg.right, -rcDlg.bottom); 
 
         // The new position is the sum of half the remaining 
         // space and the owner's original position. 
 
        SetWindowPos(hwndDlg, 
            HWND_TOP, 
            rcOwner.left + (rc.right / 2), 
            rcOwner.top + (rc.bottom / 2), 
            0, 0,          // ignores size arguments 
            SWP_NOSIZE); 
 
        if (GetDlgCtrlID((HWND) wParam) != IDCANCEL) { 
            SetFocus(GetDlgItem(hwndDlg, IDCANCEL)); 
            return FALSE; 
        } 
        return TRUE; 

    case WM_COMMAND: 
        switch (LOWORD(wParam)) {
        case IDCANCEL: 
			progressCancelled = TRUE;
            return TRUE; 
		}
	}
	return FALSE;
}

static BOOL CreateProgressDialog(HWND parentWnd, LPCTSTR title)
{
	progressDialog = CreateDialog(moduleHandle, 
						MAKEINTRESOURCE(IDD_PROGRESS_DLG),
						parentWnd, ProgressProc);
	if (progressDialog != (HWND)NULL) {
		EnableWindow(parentWnd, FALSE);
		SetWindowText(progressDialog, title);
		ShowWindow(progressDialog, SW_SHOW); 
		progressCancelled = FALSE;
		return TRUE;
	} else
		return FALSE;
}

static BOOL UpdateProgressDialog(int percent)
{
	MSG msg;
	HWND control = GetDlgItem(progressDialog, IDC_DOWNLOAD_PROGRESS);

	SendMessage(control, PBM_SETPOS, (WPARAM) percent, 0); 

	while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
		if (!IsDialogMessage(progressDialog, &msg)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}    
	}

	return !progressCancelled;
}

static BOOL DestroyProgressDialog(HWND parentWnd)
{
	EnableWindow(parentWnd, TRUE);
	DestroyWindow(progressDialog);
	return !progressCancelled;
}

// DllMain
//
// This is called when DLL is first loaded
//
BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
	moduleHandle = hModule;
	topWindowHandle = NULL;
	commHandle = NULL;
    return TRUE;
}

// IdentifyDLL
//
// The IdentifyDLL function returns an identifying string, 
// which the control program will enter in a listbox used 
// to select the appropriate DLL.
// 
// Parameters:
//    value - [out] pointer to buffer that will receive string.  
//    size  - [in] size of the buffer pointed to by value.
//
// Return Values:
//    Function returns number of bytes in returned string, 
//    if actual length of string exceeds size, the string 
//    will be truncated to size -1 bytes. 
//
// Remarks
//    The string consists of six fields, separated by the 
//    "pipe" character ("|", 0x7C), the manufacturer three letter
//    code and optional alphanumeric, the manufacturer name, 
//    supported FR name(s), DLL software revision number, and 
//    two comma separated lists of zero or more file extensions. 
//    The first list of extensions identifies manufacturer 
//    proprietary log files, if any, which can be converted to 
//    IGC format using the ConvertLog function.  The second list 
//    identifies log files (possibly including IGC format) that 
//    can be authenticated by the ValidateLog function.  A 
//    terminating NUL character is always appended to the string 
//    (but not included in the returned count).  Maximum 
//    permitted length of the string (excluding the terminating 
//    NUL character) is 127 characters.  Example:
//
//       XXX|Acme Instruments|XL 100, 200|2.0|XL1,XL2|XL1,XL2,IGC
//
DWORD __declspec(dllexport) IdentifyDLL(LPTSTR value, DWORD size)
{
	// IDENTIFY_DLL is defined at the beginning of 
	// this file.
	strncpy(value, IDENTIFY_DLL, size);
	value[size - 1] = '\0';	// make sure null-terminated
	return strlen(value);
}

// LoadIconDLL
//
// The control program calls LoadIconDLL to load a unique 32x32x4 
// (16 color) icon that may be used to identify the DLL.
//
// Return Values
//    Returns the handle for the loaded icon.  If there is an error, 
//    returns a null handle.

HICON __declspec(dllexport) LoadIconDLL()
{
	return LoadIcon(moduleHandle, MAKEINTRESOURCE(IDI_ICON));
}

// InitializeDLL
//
// InitializeDLL is an initialization function that must be 
// called before any of the other functions, with the exception 
// of IdentifyDLL or LoadIconDLL.
//
// Parameters
//    windowHandle - [in] Handle for the control programs main 
//                        window, or NULL if there is none.
//    quietMode    - [in] if TRUE, application is operating in 
//                        quiet (non-interactive) mode.
//
// Remarks
//    The window handle should be stored in the DLL, and is normally 
//    used as the parent handle for any dialog boxes displayed by DLL 
//    functions, unless overridden by a call to SetWindowDLL.  These 
//    dialogs should be centered within the parent window.  If quiet 
//    mode is requested, the DLL functions should only display dialogs 
//    resulting from non-recoverable error conditions, progress and 
//    informational dialogs should not be displayed
//
VOID __declspec(dllexport) InitializeDLL(HWND windowHandle, BOOL quietMode)
{
	topWindowHandle = windowHandle;
	quietModeFlag = quietMode;
	commHandle = INVALID_HANDLE_VALUE;
}

// SetWindowDLL
//
// The control program will call this function to set a new top-level 
// window handle.  This handle should be used as the parent window for 
// any dialog boxes displayed by DLL functions, unless overridden by 
// another call to SetWindowDLL
//
// Parameters
//    windowHandle - [in] Handle for the control programs new top level 
//                        window.
//
VOID __declspec(dllexport) SetWindowDLL(HWND windowHandle)
{
	topWindowHandle = windowHandle;
}

// KeepAwakeIntervalDLL
//
// The KeepAwakeIntervalDLL function returns the nominal time 
// interval required between calls to KeepAwakeFR.
//
// Return Values:
//    Returns the interval in milliseconds.  If 0 is returned, 
//    KeepAwakeFR calls are not required (and will be ignored).
//
DWORD __declspec(dllexport) KeepAwakeIntervalDLL()
{
	// KEEP_AWAKE_INTERVAL is defined at the beginning of 
	// this file.
	return KEEP_AWAKE_INTERVAL;
}

// UseSerialOptionsDLL
//
// The UseSerialOptionsDLL function is called by the control 
// program to determine if there are any user settable connection 
// options which may be set using the dialog provided by 
// SerialOptionsDLL.  If there are none, the control program 
// should not call SerialOptionsDLL.
//
// Return Values
//    Returns TRUE if there are user settable serial options 
//    (and SerialOptionsDLL should be called), FALSE if not.
//
BOOL __declspec(dllexport) UseSerialOptionsDLL()
{
#ifdef USE_SERIAL_OPTIONS
	return TRUE;
#else
	return FALSE;
#endif // USE_SERIAL_OPTIONS
}

// SerialOptionsDLL
//
// The SerialOptionsDLL function displays a modal dialog box 
// requesting any user settable connection options (line speed, 
// flow control, etc.) needed to configure a serial port for 
// use with the flight recorder.  
//
// Parameters
//    options - [out] pointer to buffer which will receive the 
//                    connection options.
//    size    - [in]  size of the buffer pointed to by options 
//                    in bytes.  
// Return Values
//    If successful, function returns number of bytes in the
//    returned option string. if actual length of string exceeds 
//    size, the string will be truncated to size -1 bytes. If
//    cancelled, returns 0 and options string is left unmodified.
//    If error, a modal dialog is displayed, and -1 is returned.
//
// Remarks
//    This dialog should not include selection of the serial 
//    communication device. Maximum permitted length of the 
//    returned string (excluding the terminating NUL character) 
//    is 63 characters.  The string is intended for use in a 
//    subsequent call to SerialConnectFR, the actual format of 
//    the string is determined by the manufacturer. The control 
//    program may choose to store this string in the registry or 
//    a file for use in future sessions. 
//
DWORD __declspec(dllexport) SerialOptionsDLL(LPTSTR options, DWORD size)
{
#ifdef USE_SERIAL_OPTIONS
	int err;

	//
	// Initialize and display serial options dialog
	//
	serialOptions = options;
	serialOptionsSize = size;
	err = DialogBox(moduleHandle, MAKEINTRESOURCE(IDD_OPTIONS), 
					topWindowHandle, SerialOptionsProc);
	//
	// Check result
	//
	if (err == IDOK) {
		// OK, return length of option string
		return serialOptionsSize;
	} else if (err == IDCANCEL) {
		// Cancel, return 0
		return 0;
	} else {
		// An error, report it
		LastErrorBox("Serial options");
		return -1;
	}
#else
	// Serial options not implemented
	return 0;
#endif // USE_SERIAL_OPTIONS
}

// SerialConnectFR
//
// The SerialConnectFR function is used to establish 
// communication with a FR connected through a serial 
// port.  Must be called prior to using KeepAwakeFR, 
// IdentifyFR, IdentifyLogFR, DownloadLogFR, and/or 
// DisconnectFR.
//
// Parameters
//    device  - [in] name of the serial communication device 
//                   device ("COM1", etc.).  
//    options - [in] string returned by a previous call to 
//                   SerialOptionsDLL, or NULL to use the 
//                   default device options.  The format of 
//                   this string is determined by the 
//                   manufacturer.
//
// Return Values
//    Returns TRUE if connection established, FALSE otherwise. 
//
// Remarks
//    If a connection cannot be established, the function should 
//    display a modal dialog box detailing the problem.
//
BOOL __declspec(dllexport) SerialConnectFR(LPCTSTR device, LPCTSTR options)
{
	DCB dcb;

	//
	// Open serial port
	//
	commHandle = CreateFile(device, GENERIC_READ | GENERIC_WRITE,
							0, NULL, OPEN_EXISTING, 0, NULL );
	if (commHandle == INVALID_HANDLE_VALUE) {
		// Handle the error.
		LastErrorBox(device);
		// Return failure
		return FALSE;
	}

	//
	// Get the current serial port configuration
	//
	if (!GetCommState(commHandle, &dcb)) {
		// Handle the error.
		LastErrorBox(device);
		CloseHandle(commHandle);
		commHandle = INVALID_HANDLE_VALUE;
		// Return failure
		return FALSE;
	}

	//
	// Set standard configuration: default baud rate, 
	// 8 data bits, no parity, and 1 stop bit.
	//
	dcb.BaudRate = DEFAULT_BAUD_RATE; // set the baud rate
	dcb.ByteSize = 8;             // data size, xmit, and rcv
	dcb.Parity = NOPARITY;        // no parity bit
	dcb.StopBits = ONESTOPBIT;    // one stop bit

	//
	// Option string supplied?
	//
	if (options) {
		//
		// Parse options string and set baud rate
		// TODO: modify this code if additional options required
		//
		if (strcmp(options, "4800") == 0)
			dcb.BaudRate = CBR_4800;
		else if (strcmp(options, "9600") == 0)
			dcb.BaudRate = CBR_9600;
		else if (strcmp(options, "19200") == 0)
			dcb.BaudRate = CBR_19200;
		else if (strcmp(options, "38400") == 0)
			dcb.BaudRate = CBR_38400;
		else if (strcmp(options, "57600") == 0)
			dcb.BaudRate = CBR_57600;
		else if (strcmp(options, "115200") == 0)
			dcb.BaudRate = CBR_115200;
	}

	//
	// Set final serial port configuration
	//
	if (!SetCommState(commHandle, &dcb)) {
		// Configuration failed, handle the error.
		LastErrorBox(device);
		CloseHandle(commHandle);
		commHandle = INVALID_HANDLE_VALUE;
		// Return failure
		return FALSE;
	}

	//
	// Serial port now successfully opened and configured
	//
	return TRUE;
}

// KeepAwakeFR
//
// The KeepAwakeFR function is used to prevent the FR from 
// disconnecting during idle periods between calls to ConnectFR 
// and DisconnectFR.  If KeepAwakeIntervalDLL returns a non-zero 
// value, the control program must call KeepAwakeFR each time 
// that interval elapses.
// 
// Return Values
//    Returns TRUE if the FR still connected or FALSE if 
//    connection has been broken.
//
BOOL __declspec(dllexport) KeepAwakeFR()
{
	if (CheckSerialPort(FALSE)) {
		// TODO: Change this if flight recorder will 
		//       disconnect on timeout
		return TRUE;
	} else
		return FALSE;
}

// IdentifyFR
//
// The IdentifyFR function is used to obtain the manufacturer 
// id/serial number, the FR model name/number, and the FR sealed 
// status for the connected FR. 
//
// Parameters
//    value - [out] pointer to a buffer which will receive the 
//                  string result.
//    size  - [in]  size of the buffer pointed to by value in 
//                  bytes.
//
// Returned Values
//    Function returns number of bytes in returned string, if 
//    actual length of string exceeds size, the string will be 
//    truncated to size -1 bytes.   
//
// Remarks
//    The string consists of three fields, separated by the 
//    "pipe" character ("|", 0x7C), the manufacturer id/serial 
//    number (formatted MMMNNN, where MMM is the manufacturer id,
//    and NNN is the serial number), the FR model name/number, 
//    and the FR sealed status ("SEALED" if sealed, "UNSEALED" if
//    not). Maximum permitted length of the string (excluding the
//    terminating NUL character) is 63 characters.  Example:
//
//        AXL01F|XL 100|SEALED
//
DWORD __declspec(dllexport) IdentifyFR(LPTSTR value, DWORD size)
{
	if (CheckSerialPort(TRUE)) {

		// TODO: REPLACE THIS DEMO CODE >>>

		strncpy(value, "XXX01F|XL 100|SEALED", size);
		value[size - 1] = '\0';	// make sure null-terminated
		return strlen(value);

		// <<< REPLACE THIS DEMO CODE

	} else
		return 0;
}

// IdentifyLogFR
//
// The IdentifyLogFR function is used to obtain information on 
// a log stored in the currently connected FR. 
//
// Parameters
//    index - [in]  index of the desired log, starting with 0. 
//    value - [out] pointer to the buffer which will received 
//                  the returned string.
//    size  - [in]  size of the buffer pointed to by value.
// Return Values
//    IdentifyLogFR returns number of bytes in the returned 
//    string, if actual length of string exceeds size, the 
//    string will be truncated to size -1 bytes.  If the value 
//    specified for index exceeds the number of logs present in 
//    the FR (minus 1, as indexing starts with 0), IdentifyLogFR 
//    will return 0.
//
// Remarks
//    The returned string consists of seven fields, separated by 
//    the "pipe" character ("|", 0x7C), the default log file name
//    (including extension), log start UTC date (formatted 
//    YYYY-MM-DD, example "2000-05-12", zero padding required), 
//    log start UTC time (formatted HH:MM:SS, example "17:09:22",
//    zero padding required), log end UTC time (formatted 
//    HH:MM:SS), pilot name, competition id, and competition 
//    class.  Maximum permitted length of the returned string 
//    (excluding the terminating NUL character) is 127 characters. 
//    Example:
//
//        0B8X01F1.XL1|2000-11-08|20:05:21|01:21:09|J. Doe|XYZ|15M
//
//    Logs are indexed in descending start date/time order, the 
//    log at index 0 is the most recent log. When retrieving 
//    information on all of the logs stored within the FR, a 
//    control program should start by calling IdentifyLogFR with 
//    index 0, incrementing index by 1 until IdentifyLogFR 
//    returns 0.
//
DWORD __declspec(dllexport) IdentifyLogFR(DWORD index, LPTSTR value, DWORD size)
{
	if (CheckSerialPort(TRUE)) {
		//
		// Obtain log information from flight recorder
		//

		// TODO: REPLACE THIS DEMO CODE >>>

		// Return a information on a dummy log at index 0,
		// return nothing for other indices.
		if (index == 0) {
			strncpy(value, "0B8X01F1.XL1|2000-11-08|20:05:21|01:21:09|J. Doe|XYZ|15M", size);
			value[size - 1] = '\0';	// make sure null-terminated
		} else
			value[0] = '\0';
		return strlen(value);

		// <<< REPLACE THIS DEMO CODE
	} else
		return 0;
}

// DownloadLogFR
//
// The DownloadLogFR function is used to download a log file 
// from the currently connected FR.
//
// Parameters
//    index    - [in] the index of the desired log, starting 
//                    with 0. 
//    fileName - [in] a null terminated string containing the 
//                    name of the file (which may include a path)
//                    to which the log will be downloaded.  If 
//                    NULL, the default file name will be used 
//                    in the current working directory.
//
// Return Values
//    DownloadLogFR returns TRUE if successful, FALSE if there 
//    was an error.
//
// Remarks
//    If a file with the specified name and path already existed,
//    it will be overwritten. If there is an error, DownloadLogFR
//    should display a modal dialog box giving the details.
//
BOOL __declspec(dllexport) DownloadLogFR(DWORD index, LPCTSTR filename)
{
	HANDLE hFile;
	DWORD  writeCount;

	if (CheckSerialPort(TRUE)) {
		LPCTSTR fileTitle;
		TCHAR progressTitle[MAX_PATH];
		BOOL status = TRUE;
		int completed;

		//
		// Check that log index is in valid range,
		// and obtain header information
		//

		// TODO: REPLACE THIS DEMO CODE >>>

		if (index != 0) {	// dummy log at index 0 only
			ErrorBox("Flight log index out of range");
			return FALSE;
		}
		// If no filename specified, use dummy default name
		if (filename == NULL)
			filename = "0B8X01F1.XL1";

		// <<< REPLACE THIS DEMO CODE

		//
		// Create log file
		//

		hFile = CreateFile(filename, GENERIC_WRITE, 0, NULL,  
						   CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);   
		// Check that file successfully created
		if (hFile == INVALID_HANDLE_VALUE) { 
			LastErrorBox(filename);  // process error
			return FALSE;
		}

		//
		// Download log and write file
		//

		if (!quietModeFlag) {
			strcpy(progressTitle, "Downloading ");
			fileTitle = strrchr(filename, '\\');
			strcat(progressTitle, (fileTitle ? fileTitle + 1 : filename));
			strcat(progressTitle, "...");
			CreateProgressDialog(topWindowHandle, progressTitle);
		}

		// TODO: REPLACE THIS DEMO CODE >>>

		for (completed = 0; completed < 100; completed++) {
			if (!quietModeFlag && !UpdateProgressDialog(completed))
				break;
#define DEMO_DOWNLOAD "This is a demo flight log\n"
			WriteFile(hFile, DEMO_DOWNLOAD, sizeof(DEMO_DOWNLOAD), 
					  &writeCount, NULL);
			Sleep(50);
		}

		// <<< REPLACE THIS DEMO CODE

		if (!quietModeFlag)
			status = DestroyProgressDialog(topWindowHandle);


		//
		// Close log file
		//

		CloseHandle(hFile);
		return status;
	} else
		return FALSE;
}

// DisconnectFR
//
// DisconnectFR is called after the control program has completed
// interaction with the FR, to close the communication device.
//
VOID __declspec(dllexport) DisconnectFR()
{
	//
	// Close serial port
	//
	if (commHandle != INVALID_HANDLE_VALUE) {
		CloseHandle(commHandle);
		commHandle = INVALID_HANDLE_VALUE;
	}
}

// UseConvertLog
//
// The UseConvertLog function is called by the control program to 
// determine if the DLL ConvertLog function is needed to convert 
// from a proprietary log file format to IGC.  If it does not, the 
// control program should not offer the user the option of converting 
// log files.
// 
// Return Values
//    Returns TRUE if conversion from proprietary format to IGC is 
//    required, FALSE if not
//
BOOL __declspec(dllexport) UseConvertLog()
{
#ifdef USE_CONVERT_LOG
	return TRUE;
#else
	return FALSE;
#endif // USE_CONVERT_LOG
}

// ConvertLog [OPTIONAL]
//
// ConvertLog converts the log file specified by fileName to an 
// IGC format file specified by igcFileName.
//
// Parameters
//    fileName    - [in] a null terminated string containing the 
//                       name of an existing log file (which may 
//                       include a path) in the manufacturer 
//                       proprietary format.
//    igcFileName - [in] a null terminated string containing the 
//                       name of the IGC file (which may include 
//                       a path) to be created.
// Return Values
//    Returns TRUE if successful, FALSE if there is an error. 
//
// Remarks
//    This is an optional function, and should only be exported 
//    by the DLL if log files are downloaded in a non-IGC 
//    proprietary format.  If a file with the specified 
//    igcFileName already exists, it will be overwritten.  If 
//    there is an error, the function should display a modal 
//    dialog with the details.
//
BOOL __declspec(dllexport) ConvertLog(LPCTSTR filename, LPCTSTR igcFilename)
{
#ifdef USE_CONVERT_LOG
	HANDLE hFile, hFileOut;
	LPCTSTR fileTitle;
	TCHAR progressTitle[MAX_PATH];
	BOOL status = TRUE;
	int completed;

	//
	// Open proprietary log file
	//
	hFile = CreateFile(filename, GENERIC_READ, 0, NULL,  
					   OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	// Check that file successfully opened
	if (hFile == INVALID_HANDLE_VALUE) { 
		LastErrorBox(filename);  // process error
		return FALSE;
	}

	//
	// Create IGC file
	//
	hFileOut = CreateFile(igcFilename, GENERIC_WRITE, 0, NULL,
						  CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL); 
	// Check that file successfully created
	if (hFileOut == INVALID_HANDLE_VALUE) { 
		LastErrorBox(igcFilename);  // process error
		CloseHandle(hFile);
		return FALSE;
	}
	//
	// Convert proprietary log to IGC format
	//

	if (!quietModeFlag) {
		strcpy(progressTitle, "Converting ");
		fileTitle = strrchr(filename, '\\');
		strcat(progressTitle, (fileTitle ? fileTitle + 1 : filename));
		strcat(progressTitle, "...");
		CreateProgressDialog(topWindowHandle, progressTitle);
	}

	// TODO: REPLACE THIS DEMO CODE >>>

	for (completed = 0; completed < 100; completed++) {
		if (!quietModeFlag && !UpdateProgressDialog(completed))
			break;
		Sleep(50);
	}

	// <<< REPLACE THIS DEMO CODE

	if (!quietModeFlag)
		status = DestroyProgressDialog(topWindowHandle);

	//
	// Close file handles and return TRUE
	//
	CloseHandle(hFile);
	CloseHandle(hFileOut);
	return TRUE;
#else
	return FALSE;
#endif // USE_CONVERT_LOG
}

// ValidateLog
//
// ValidateLog is called to authenticate the digital signature 
// on a specified log file.
//
// Parameters
//    fileName - [in] a null terminated string containing the 
//                    name of an existing log file (which may 
//                    include a path) to be validated.
//
// Return Values
//    Returns TRUE if file can be validated, FALSE otherwise.
//
// Remarks
//    If the log was not produced by a supported flight recorder,
//    is in an unsupported format, or the digital signature is 
//    invalid, the function should display a modal dialog 
//    detailing the problem, then return FALSE.
//
BOOL __declspec(dllexport) ValidateLog(LPCTSTR filename)
{
	HANDLE hFile;
	LPCTSTR fileTitle;
	TCHAR progressTitle[MAX_PATH];
	BOOL status = TRUE;
	int completed;

	//
	// Open log file
	//
	hFile = CreateFile(filename, GENERIC_READ, 0, NULL,  
					   OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL); 
	// Check that file successfully opened
	if (hFile == INVALID_HANDLE_VALUE) { 
		LastErrorBox(filename);		  // process error
		return FALSE;
	}

	//
	// Validate log file signature
	//

	if (!quietModeFlag) {
		strcpy(progressTitle, "Validating ");
		fileTitle = strrchr(filename, '\\');
		strcat(progressTitle, (fileTitle ? fileTitle + 1 : filename));
		strcat(progressTitle, "...");
		CreateProgressDialog(topWindowHandle, progressTitle);
	}

	// TODO: REPLACE THIS DEMO CODE >>>

	for (completed = 0; completed < 100; completed++) {
		if (!quietModeFlag && !UpdateProgressDialog(completed))
			break;
		Sleep(50);
	}

	// <<< REPLACE THIS DEMO CODE

	if (!quietModeFlag)
		status = DestroyProgressDialog(topWindowHandle);

	//
	// Close file handle and return TRUE
	//
	CloseHandle(hFile);
	return status;
}
